"use client";

import axios from "axios";
import { get, isEmpty } from "lodash";
import { useEffect, useRef, useState } from "react";

import { LIST_DATA_DATE_TIME_FORMAT } from "@/common/constant";
import FilterModal from "@/components/Modal/AttendanceModal/FilterModal";
import Modal from "@/components/Modal/Modal";
import ListHeader from "@/components/Shared/ListHeader";
import useSearchQuery from "@/hooks/useSearchQuery";
import ListWithAverage from "@/shared/ListWithAverage";
import withAuth from "@/shared/withAuth";
import { getFormattedDate } from "@/utilities/dateHelper";
import { EnumHelper } from "@/utilities/EnumHelper/enumHelper";
import { timeDifferenceHelper } from "@/utilities/timeDifferenceHelper";
import { toastError } from "@/utilities/Toast/toastHelper";
import { useSearchParams } from "next/navigation";
import { FaFilePdf, FaFilter } from "react-icons/fa";

const DISPLAY = {
  title: () => "KPI Report",
  content: {
    properties: [
      "id",
      "employee_name",
      "check_in",
      "check_out",
      "duration",
      "average_entry",
      "average_exit",
      "average_duration",
    ],
    headerClass: {},
    bodyClass: {},
    style: {
      columnWidth: "md:grid-cols-[0.3fr_0.3fr_2.5fr_2fr_2fr_1fr_1fr_1fr_1fr]", // 1st 1fr for "SL" (if autoSerialNumber true)
      printColumnWidth:
        "print:grid-cols-[0.3fr_0.3fr_2.5fr_2fr_2fr_1fr_1fr_1fr_1fr]", // 1st 1fr for "SL" (if autoSerialNumber true)
    },
    header: () => {
      return {
        id: "Id",
        employee_name: "Employee Name",
        check_in: "Check In",
        check_out: "Check Out",
        duration: "Duration",
        average_entry: "Avg Entry",
        average_exit: "Avg Exit",
        average_duration: "Avg Duration",
      };
    },
    body: ({ row, column }) => {
      if (column === "id") {
        const id = get(row, "emp_code", "");
        return id ? id : "-";
      }

      if (column === "employee_name") {
        const id = get(row, "emp_code", "");
        return `${EnumHelper({
          type: "employeeInfo",
          value: id,
          display: true,
        })}`;
      }

      if (column === "check_in") {
        const check_in = get(row, "check_in", "");
        return (
          <span className="text-xs">
            {check_in
              ? getFormattedDate(check_in, LIST_DATA_DATE_TIME_FORMAT)
              : "-"}
          </span>
        );
      }

      if (column === "check_out") {
        const check_out = get(row, "check_out", "");
        return (
          <span className="text-xs">
            {check_out
              ? getFormattedDate(check_out, LIST_DATA_DATE_TIME_FORMAT)
              : "-"}
          </span>
        );
      }

      if (column === "duration") {
        const check_in = get(row, "check_in", "");
        const check_out = get(row, "check_out", "");
        const result = timeDifferenceHelper(check_out, check_in);
        return (
          <span>
            {get(result, "hours", 0)}:{get(result, "minutes", 0)}h
          </span>
        );
      }

      if (column === "average_entry") {
        const average_entry = get(row, "average_entry", "");
        const time = average_entry.split(":");
        const hours = time[0];
        const minutes = time[1];
        const ampm = hours >= 12 ? "PM" : "AM";
        const hour = hours % 12 || 12;
        return (
          average_entry && (
            <span className="text-xs">{`${hour}:${minutes} ${ampm}`}</span>
          )
        );
      }

      if (column === "average_exit") {
        const average_exit = get(row, "average_exit", "");
        const time = average_exit.split(":");
        const hours = time[0];
        const minutes = time[1];
        const ampm = hours >= 12 ? "PM" : "AM";
        const hour = hours % 12 || 12;
        return (
          average_exit && (
            <span className="text-xs">{`${hour}:${minutes} ${ampm}`}</span>
          )
        );
      }

      if (column === "average_duration") {
        const average_duration = get(row, "average_working_hours", "");
        return <span className="text-xs">{average_duration}</span>;
      }

      const typeOfData = typeof row[column];
      return typeOfData === "string"
        ? row[column]
          ? row[column]
          : "-"
        : typeOfData === "number"
        ? row[column]
        : "-";
    },
    customColumnClassNames: [
      {
        property: "action",
        className: "md:text-right flex justify-end print:hidden",
      },
    ],
  },
};

function KPIReport() {
  const [isLoading, setIsLoading] = useState(true);
  const [attendanceData, setAttendanceData] = useState({});
  const [filterModalOpen, setFilterModalOpen] = useState(false);

  const params = useSearchParams();
  const searchParams = useSearchQuery(params);

  const printRef = useRef();
  const printFunctionRef = useRef(null);

  const title = DISPLAY.title();

  const updateAttendanceData = (data) => {
    setAttendanceData(data);
    setIsLoading(false);
  };

  const onCloseFilterModal = () => {
    setFilterModalOpen(false);
  };

  const fetchEmployeeRecords = (params = searchParams, search = false) => {
    const data = axios
      .get("https://zkteco.atpldhaka.com/api/employee-records", {
        ...(search && { params }),
      })
      .then((response) => {
        const data = get(response, "data", {});
        updateAttendanceData(data);
        setIsLoading(false);
      })
      .catch((error) => {
        toastError({ message: "Something went wrong!" });
        setIsLoading(false);
      })
      .finally(() => {
        setIsLoading(false);
      });

    return data;
  };

  const onClickPdf = () => {
    printFunctionRef.current();
  };

  const init = () => {
    setIsLoading(true);
    if (isEmpty(searchParams)) {
      fetchEmployeeRecords();
      return;
    } else {
      fetchEmployeeRecords(searchParams, true);
    }
  };
  useEffect(() => {
    fetchEmployeeRecords(searchParams, true);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  return (
    <div className="" ref={printRef}>
      <title>Attendance ATPL-AMS</title>
      <ListHeader title={title}>
        <button
          className="print:hidden bg-tertiary h-10 p-1 pt-2 md:w-8 md:h-8 w-full flex justify-center items-center rounded-full text-white hover:text-gray-200"
          onClick={() => setFilterModalOpen(true)}
        >
          <FaFilter className="fixed" size={16} />
          {(searchParams?.check_in_start ||
            searchParams?.check_in_end ||
            searchParams?.emp_code) && (
            <span class="relative -top-3 -right-3 flex h-2 w-2">
              <span class="animate-ping absolute inline-flex h-full w-full rounded-full bg-blue-800 opacity-75"></span>
              <span class="relative inline-flex rounded-full h-2 w-2 bg-sidebar"></span>
            </span>
          )}
        </button>
        <button
          className="print:hidden bg-tertiary h-10 p-1 md:w-8 md:h-8 w-full flex justify-center items-center rounded-full text-white hover:text-gray-200"
          onClick={() => onClickPdf()}
        >
          <FaFilePdf className="fixed" size={16} />
        </button>
      </ListHeader>
      <ListWithAverage
        printFunctionRef={printFunctionRef}
        title={title}
        loading={isLoading}
        data={attendanceData}
        renderDropdownItem={false}
        queryParams={searchParams}
        onChangeData={updateAttendanceData}
        properties={DISPLAY.content.properties}
        header={DISPLAY.content.header()}
        body={({ row, column }) =>
          DISPLAY.content.body({
            row,
            column,
          })
        }
        style={DISPLAY.content.style}
        customColumnClassNames={DISPLAY.content.customColumnClassNames}
        callApi={init}
      />
      <Modal
        isOpen={filterModalOpen}
        onClose={onCloseFilterModal}
        title="Filter Attendance"
        size="sm"
      >
        <FilterModal
          onClose={onCloseFilterModal}
          type="add"
          init={fetchEmployeeRecords}
        />
      </Modal>
    </div>
  );
}
export default withAuth(KPIReport);
