import { LIST_DATA_DATE_FORMAT } from "@/common/constant";
import DefaultButton from "@/components/Button/DefaultButton";
import SubmitButton from "@/components/Button/SubmitButton";
import Select from "@/components/Input/Select";
import useSearchQuery from "@/hooks/useSearchQuery";
import { getFormattedDate } from "@/utilities/dateHelper";
import { EnumHelper } from "@/utilities/EnumHelper/enumHelper";
import {
  Popover,
  PopoverButton,
  PopoverPanel,
  Transition,
} from "@headlessui/react";

import { usePathname, useRouter, useSearchParams } from "next/navigation";
import { Fragment, useEffect, useMemo, useState } from "react";
import { DateRangePicker } from "react-date-range";

const initialState = {
  check_in_start: "",
  check_in_end: "",
  emp_code: "",
};

function FilterModal({ init = () => {}, onClose = () => {} }) {
  const [formData, setFormData] = useState({ ...initialState });
  const [state, setState] = useState([
    {
      startDate: new Date(),
      endDate: new Date(),
      key: "selection",
    },
  ]);

  const params = useSearchParams();
  const searchParams = useSearchQuery(params);
  const router = useRouter();
  const path = usePathname();

  const handleRegister = (e) => {
    e.preventDefault();
    e.stopPropagation();
    const params = {
      ...(formData?.emp_code ? { emp_code: formData?.emp_code } : {}),
      ...(formData?.check_in_start
        ? {
            check_in_start: getFormattedDate(
              formData?.check_in_start,
              "yyyy-MM-dd"
            ),
          }
        : {}),
      ...(formData?.check_in_end
        ? {
            check_in_end: getFormattedDate(
              formData?.check_in_end,
              "yyyy-MM-dd"
            ),
          }
        : {}),
    };
    router.push(`${path}?${new URLSearchParams(params).toString()}`);
    init(params, true);
    onClose();
  };

  const employeeOptions = useMemo(() => {
    return EnumHelper({ type: "employeeInfo", options: true });
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const getSelectValue = (options, value) => {
    return options.filter((item) => item.value === value);
  };

  const onClickClose = () => {
    // clear all params
    router.push(path);
    init();
    onClose();
  };

  useEffect(() => {
    if (searchParams) {
      setFormData({
        emp_code: searchParams?.emp_code,
        check_in_start: searchParams?.check_in_start,
        check_in_end: searchParams?.check_in_end,
      });
      setState([
        {
          startDate: searchParams?.check_in_start
            ? new Date(searchParams?.check_in_start)
            : new Date(),
          endDate: searchParams?.check_in_end
            ? new Date(searchParams?.check_in_end)
            : new Date(),
          key: "selection",
        },
      ]);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  return (
    <div className="">
      <form action="submit" onSubmit={handleRegister} className="">
        <div className="mb-4">
          <Popover className="relative">
            <PopoverButton className="w-full">
              <label
                htmlFor="email"
                className="block text-sm font-semibold text-gray-800 text-left"
              >
                Date Range
              </label>
              <input
                type="text"
                id="text"
                placeholder="Date Range"
                onChange={() => {}}
                value={
                  formData?.check_in_start && formData?.check_in_end
                    ? `${getFormattedDate(
                        formData?.check_in_start,
                        LIST_DATA_DATE_FORMAT
                      )} - ${getFormattedDate(
                        formData?.check_in_end,
                        LIST_DATA_DATE_FORMAT
                      )}`
                    : ""
                }
                className="block w-full px-2 py-2 mt-2 text-gray-700 bg-white border rounded-md focus:border-gray-400 focus:ring-gray-300 focus:outline-none focus:ring focus:ring-opacity-40"
              />
            </PopoverButton>
            <Transition
              as={Fragment}
              enter="transition ease-out duration-200"
              enterFrom="opacity-0 translate-y-1"
              enterTo="opacity-100 translate-y-0"
              leave="transition ease-in duration-150"
              leaveFrom="opacity-100 translate-y-0"
              leaveTo="opacity-0 translate-y-1"
            >
              <PopoverPanel className="absolute z-10 rounded-lg">
                <DateRangePicker
                  onChange={(item) => {
                    setState([item.selection]);
                    setFormData({
                      ...formData,
                      check_in_start: item.selection.startDate,
                      check_in_end: item.selection.endDate,
                    });
                  }}
                  showSelectionPreview={true}
                  moveRangeOnFirstSelection={false}
                  months={1}
                  ranges={state}
                  direction="horizontal"
                  className="custom-date-range"
                />
              </PopoverPanel>
            </Transition>
          </Popover>
        </div>
        <div className="mb-4">
          <label
            htmlFor="email"
            className="block text-sm font-semibold text-gray-800"
          >
            Employee Code
          </label>
          <Select
            options={employeeOptions}
            value={getSelectValue(employeeOptions, formData?.emp_code)}
            placeholder={"Employee Code"}
            closeMenuOnSelect
            name="employee_code"
            onChange={(item) => {
              setFormData({
                ...formData,
                emp_code: item?.value,
              });
            }}
          />
        </div>
        <div className="flex flex-col-reverse md:flex-row gap-4 justify-between items-center">
          <DefaultButton
            onClick={onClickClose}
            type="reset"
            className="w-full bg-warning"
          >
            Reset
          </DefaultButton>
          <SubmitButton className="w-full bg-tertiary">Apply</SubmitButton>
        </div>
      </form>
    </div>
  );
}

export default FilterModal;
